% main script for load flow simulation

clear all;
close all;

disp(' ');
disp('********************************************************');
disp('* This script solves compares different load flow approximations *');
disp('********************************************************');
disp(' ');

addpath(genpath('CodeY'))
%%  Step 1: load Y matrix & base values
%   The nodal admittance matrix should saved in a folder 'Data_LF'
%   located in the same directory as this script.

disp('STEP 1: computing nodal admittance matrix and base values ...');
disp(' ');

% ! Define the network to use !
network = 'high voltage'; % either 'high voltage' or 'medium voltage' or 'low voltage'

% choose correct .mat files
switch(network)
    case 'high voltage'
        text_file  = './Data_LF/HV Network/data_lines_HV.txt';
        Ab_file = './Data_LF/HV Network/Base_Power.mat';
        Vb_file = './Data_LF/HV Network/Base_Voltage.mat';
    case 'medium voltage'
        text_file  = './Data_LF/MV Network/data_lines_MV.txt';
        Ab_file = './Data_LF/MV Network/Base_Power.mat';
        Vb_file = './Data_LF/MV Network/Base_Voltage.mat';   
    case 'low voltage'
        text_file  = './Data_LF/LV Network/data_lines_LV.txt';
        Ab_file = './Data_LF/LV Network/Base_Power.mat';
        Vb_file = './Data_LF/LV Network/Base_Voltage.mat';  
end


% Put the names of the files containing the base values
Ab = importdata(Ab_file);
Vb = importdata(Vb_file);

% Put the name of the file containing the Y matrix
[Y, linedata] = computeY(text_file, Ab, Vb);

n_nodes = size(Y,1); % number of nodes

disp(['The network consists of ',num2str(n_nodes), ' nodes.']);
disp(['The base power is ' num2str(Ab/1e6), 'MVA.']);
disp(['The base voltage is ' num2str(Vb/1e3) 'kV.']);
disp(' ');

%%  Step 2: load power profiles of loads and generators
% The nodal power profiles should be also be saved 'Data_LF',
% as matrices of size n_nodes x n_timesteps.
disp('STEP 2: Loading power profiles.')
disp(' ');
profile_type = 'daily';
switch(network)
    case 'low voltage'
        P_load = importdata('./Data_LF/LV Network/P_daily_load_curve.mat');
        Q_load = importdata('./Data_LF/LV Network/Q_daily_load_curve.mat');

        P_gen = importdata('./Data_LF/LV Network/P_daily_gen_curve.mat');
        Q_gen = importdata('./Data_LF/LV Network/Q_daily_gen_curve.mat');

     case 'medium voltage'
        P_load = importdata('./Data_LF/MV Network/P_daily_load_curve.mat');
        Q_load = importdata('./Data_LF/MV Network/Q_daily_load_curve.mat');

        P_gen = importdata('./Data_LF/MV Network/P_daily_gen_curve.mat');
        Q_gen = importdata('./Data_LF/MV Network/Q_daily_gen_curve.mat');

    case'high voltage'
        switch profile_type
            case 'daily'
                P_load = importdata('./Data_LF/HV Network/P_daily_load_curve.mat');
                Q_load = importdata('./Data_LF/HV Network/Q_daily_load_curve.mat');

                P_gen = importdata('./Data_LF/HV Network/P_daily_gen_curve.mat');
                Q_gen = importdata('./Data_LF/HV Network/Q_daily_gen_curve.mat');
                
            case 'single'
                P_load = importdata('./Data_LF/HV Network/P_single_load.mat');
                Q_load = importdata('./Data_LF/HV Network/Q_single_load.mat');

                P_gen = importdata('./Data_LF/HV Network/P_single_gen.mat');
                Q_gen = importdata('./Data_LF/HV Network/Q_single_gen.mat');
                
                
         end
    otherwise
        error('unknown network type');
end

% sanity check
if (~all([size(P_load,1),size(Q_load,1),size(P_gen,1),size(Q_gen,1)]==n_nodes))
    error('The P/Q profiles are not compatible in terms of number of nodes.');
elseif (length(unique([size(P_load,2),size(Q_load,2),size(P_gen,2),size(Q_gen,2)]))~=1)
    error('The P/Q profiles are not compatible in terms of number of timesteps.');
end

% Complex power in p.u. (net)
S_star = 1*complex(P_gen-P_load,Q_gen-Q_load) / Ab;

n_timesteps = size(S_star,2);

disp(['The number of timesteps is ' num2str(n_timesteps) '.']);
disp(' ');

%%  Step 3: Simulation parameters
%%% Set the complex load values in p.u. and initialize the bus voltages
%%% Also define tolerance and max number of allowed iterations for the NR

disp('STEP 3: Configuring the load flow simulation.');
disp(' ');

% Configure the bus types
% index of the slack bus
idx.slack = 1;
% indices of the PV buses
idx.pv = [];
% indices of the PQ buses
idx.pq = (2:n_nodes)';

% ! Customize the load flow simulation !

% Choose LF solver  'wardhale' or 'carpentier' or 'stott' when
% 'profile_type = 'daily'
LF_solver = 'stott';

% ! Enter Newton-Raphson algorithm parameters !

% maximum number of iterations
Parameters.n_max = 100;
% tolerance for the convergence criterion
Parameters.tol = 1e-7;

%% Step 4: Load flow approximation

disp('STEP 4: running Newton-Raphson algorithm.');
disp(' ');

% initialize
E_star = zeros(size(S_star)); E_star(idx.pv,:) = 1; % we assume here that all PV nodes have 1 p.u. as voltage reference
E = zeros(n_nodes,n_timesteps);
S = zeros(n_nodes,n_timesteps);
n_iter = zeros(1,n_timesteps);
t_exec = zeros(1,n_timesteps);

for k = 1:n_timesteps
    %% Initialization

    if(k<=1)
        E_0 = ones(n_nodes,1);
    else
        E_0 = E(:,k-1);
    end

    %% Call Newton-Raphson function
    
    t_start = tic;
    
    switch(LF_solver)
        
        case 'wardhale'
            [J,E(:,k),S(:,k),n_iter(k)] = NR_wardhale(S_star(:,k),E_star(:,k),Y,E_0,idx,Parameters);
        case 'carpentier'
            [J,E(:,k),S(:,k),n_iter(k)] = NR_carpentier(S_star(:,k),E_star(:,k),Y,E_0,idx,Parameters);
        case 'stott'
            [J,E(:,k),S(:,k),n_iter(k)] = NR_stott(S_star(:,k),E_star(:,k),Y,E_0,idx,Parameters);
     end
    
    
    t_exec(k) = toc(t_start);
    
end

%% Step 5: Comparison with the true values
% initialize
E_star = zeros(size(S_star)); E_star(idx.pv,:) = 1; % we assume here that all PV nodes have 1 p.u. as voltage reference
E_true = zeros(n_nodes,n_timesteps);
S_true = zeros(n_nodes,n_timesteps);
n_iter_true = zeros(1,n_timesteps);
t_exec_true = zeros(1,n_timesteps);

for k = 1:n_timesteps
    %% Initialization

    if(k<=1)
        E_0 = ones(n_nodes,1);
    else
        E_0 = E_true(:,k-1);
    end

    %% Call Newton-Raphson function
    
    t_start = tic;

            [J,E_true(:,k),S_true(:,k),n_iter_true(k)] = NR_polar(S_star(:,k),E_star(:,k),Y,E_0,idx,Parameters);
    
    t_exec_true(k) = toc(t_start);
    
end

%% Step 6: Visualization and metrics
switch profile_type
    case 'daily'
% Profiles over a day
Time = ((1:size(S_star,2)).' - 1)*24/96;
figure(2);
clf;
        
subplot(2,2,1);
plot(Time, real(S)');
title('Active Power (p.u.)'); xlabel('Time [h]');
grid on; axis tight;

subplot(2,2,2);
plot(Time, imag(S)');
title('Reactive Power (p.u.)'); xlabel('Time [h]');
grid on; axis tight;

subplot(2,2,3);
plot(Time, abs(E)');
title('Voltage Magnitude (p.u.)'); xlabel('Time [h]');
grid on; axis tight;

subplot(2,2,4);
plot(Time, rad2deg(angle(E))');
title('Phase Angle (deg)'); xlabel('Time [h]');
grid on; axis tight;

mean_iter = mean(n_iter);
max_iter = max(n_iter);

disp(['Mean and max iteration for ', LF_solver, ' are ' num2str(mean_iter), ' and ', num2str(max_iter), '.'])

% Performance

figure(3);
clf;

subplot(1,2,1);
plot(n_iter');
title('Number of Iterations');
grid on;

subplot(1,2,2);
plot(t_exec');
title('Execution Time (s)');
grid on;
    case 'single'
        disp('Voltage phasors using DC flow are')
        disp(E)
        disp(' ')
        
        disp('Voltage magnitude (pu) and phase angles (deg.) using DC flow are')
        disp([abs(E) rad2deg(angle(E))])
        disp(' ')        
        
        
        disp('True voltage phasors are')
        disp(E_true)
        
        disp('True voltage magnitudes (pu) and phase angles (deg.) are')
        disp([abs(E_true) rad2deg(angle(E_true))])
    otherwise 
        
end

% compute the error on the magnitude abd phase of the nodal voltages
% LF metrics
% - error on voltage magnitudes
V_mag_error = (abs(E) - abs(E_true));
V_mag_error_rmse = norm(V_mag_error(:))/n_timesteps;
V_mag_error_max = max(abs(V_mag_error(:)));

disp(['Rmse and max-absolute error on magnitude of voltage (in pu) are ' , num2str(V_mag_error_rmse), ' and ',  num2str(V_mag_error_max)])

% - error on voltage phase angles
Phase_error = rad2deg((angle(E) - angle(E_true)));
Phase_error_rmse = norm(Phase_error(:))/n_timesteps;
Phase_error_max = max(abs(Phase_error(:)));

disp(['Rmse and max-absolute error on phase (in deg.) of voltage are ' , num2str(Phase_error_rmse), ' and ',  num2str(Phase_error_max)])

